  program crea_3D_grids
  
! -----------------------------------------------------------------------------
! Frame   : Procedures and program used in JGR data report 
!           "Use of twenty years CLUSTER/FGM data to observe the mean behavior 
!           of the magnetic field and current density of Earth's magnetosphere" 
!           By P. Robert and M. Dunlop, October 2021
!
! Program : crea_3D_grids.f90
! Object  : Compute 3 3D grid from a list of Curl files (over 20 years)
!         : grid B, grid J, grid N
! Input   : file containing path of all Curl files over 20 years data
! Output  : 3D grid files for B, J and N
! Author  : P. Robert, LPP-ScientiDev, 2020-Oct.2021
! Mail    : patrick.robert@lpp.polytechnique.fr
! -----------------------------------------------------------------------------

  real(kind=4), dimension(4) :: Px,Py,Pz,Bx,By,Bz,  Pxe,Pye,Pze,Bxe,Bye,Bze
  character(len=255) :: file_curl_div,datafileB,datafileJ,datafileN

  real(kind=4) :: Jxx,Jyy,Jzz,J00

  character(len=3)   :: cdta
  character(len=24)  :: datiso,dirgrid

  real(kind=4), dimension(:,:,:,:,:), allocatable :: gridB,gridJ
  integer,      dimension(:,:,:,:),   allocatable :: nvalB,nvalJ
  real(kind=4), dimension(3,4) :: satpos

  x1=-20.
  x2= 20.
  y1=-20.
  y2= 20.
  z1=-20.
  z2= 20.

  dx= 0.5
  dy= 0.5
  dz= 0.5

  nx= int((x2-x1)/dx +0.5)
  ny= int((y2-y1)/dy +0.5)
  nz= int((z2-z1)/dz +0.5)

  print*, 'nx,ny,nz=',nx,ny,nz

  deltax=float(nx)*dx
  deltay=float(ny)*dy
  deltaz=float(nz)*dz

  difx=(x2-x1)-deltax
  if(abs(difx) > 1.e-10) then
     print*, 'x2-x1 =',x2-x1
     print*, 'deltax=',deltax
     print*, 'diff  =',difx
     print*, '***incoherence! aborted!!'
     stop '***incoherence! aborted!!'
  endif

  dify=(y2-y1)-deltay
  if(abs(dify) > 1.e-10) then
     print*, 'y2-y1 =',y2-y1
     print*, 'deltay=',deltay
     print*, 'diff  =',dify
     print*, '***incoherence! aborted!!'
     stop '***incoherence! aborted!!'
  endif

  difz=(z2-z1)-deltaz
  if(abs(difz) > 1.e-10) then
     print*, 'z2-z1 =',z2-z1
     print*, 'deltaz=',deltaz
     print*, 'diff  =',difz
     print*, '***incoherence! aborted!!'
     stop '***incoherence! aborted!!'
  endif
  
! test if directory grid_results exist

  dirgrid='./grid_results'  
  open(7,file=trim(dirgrid)//'/toto.tmp',iostat=iosta)
  if(iosta/=0) then
     stop '*** grid_results directory does not exist'
               else
     close(7)
  endif

  allocate(gridB(4,nx,ny,nz,9))
  allocate(gridJ(7,nx,ny,nz,9))

  allocate(nvalB(nx,ny,nz,9))
  allocate(nvalJ(nx,ny,nz,9))
  
  gridB(:,:,:,:,:)=0.
  gridJ(:,:,:,:,:)=0.
  
  nvalB(:,:,:,:)=0
  nvalJ(:,:,:,:)=0

! ---------------
! read files Curl_Div_*.dat
! ---------------

  iosta=0
  nt=0
  nf=0
  igx=0
  igy=0
  igz=0
  
  RE=6372.2
  
  print*
  print*,'-------------------------------------'
  print*,' 1) reading all Curl_Div files...'
  print*,'-------------------------------------'

  DO while (iosta == 0)
  
  read(*,'(a)', iostat=iosta) file_curl_div
  if(iosta /= 0) go to 30
  if(file_curl_div(1:5) == '     ') go to 30

  open(1, file=file_curl_div)
  
  read(1,*,iostat=iosta) nbloc
  print*
  print 100, 'read ',trim(file_curl_div),'  nbloc=',nbloc
  nf=nf+1
  nr=0
  
  100 format(3a,i8)
  200 format(a24,/,12E14.6,/,12E14.6,/,6E11.3,2E10.2,E11.3)
  
! lecture de chaque enregistrement

  DO n=1,nbloc
  
  read(1,200,iostat=iosta2,end=20) datiso,Pxe,Pye,Pze,Bxe,Bye,Bze,GJxe,GJye,GJze,curl,div,ajb,elong,plana,diptan 


  if(iosta2 /= 0) go to 20
      
  nt=nt+1
  nr=nr+1    
  
! passage en RE

  Pxe(:)=Pxe(:)/RE
  Pye(:)=Pye(:)/RE
  Pze(:)=Pze(:)/RE

! passage en gsm

  call decode_datiso(datiso,iyear,imon,iday,ih,im,is,ims,imc) 
  call ctimpar(iyear,imon,iday,ih,im,is)
      
  do isat=1,4    
      call t_gse_to_gsm(Pxe(isat),Pye(isat),Pze(isat),Px(isat),Py(isat),Pz(isat))
      call t_gse_to_gsm(Bxe(isat),Bye(isat),Bze(isat),Bx(isat),By(isat),Bz(isat))
  enddo
  
  call t_gse_to_gsm(GJxe,GJye,GJze,GJx,GJy,GJz)

  satpos(1,:)=Pxe(:)
  satpos(2,:)=Pye(:)
  satpos(3,:)=Pze(:)
  
  call codist(satpos,dmin,dmax)

! calcul de l'indice du dipole  
  
  if(diptan > 0) then
     kdta= int(( diptan+5.)/10.)
                 else
     kdta=-int((-diptan+5.)/10.)
  endif
  idta=kdta+5  ! varie entre 1 et 9 pour dta entre -35 et 35
  nbdta=9
  
  if(idta < 1 .or. idta > nbdta) cycle


! calcul des grilles
! ------------------  
  
  do iclu=1,4

     igx=int((Px(iclu)-x1)/dx) +1
     igy=int((Py(iclu)-y1)/dy) +1
     igz=int((Pz(iclu)-z1)/dz) +1

     if(igx < 1 .or. igx > nx) cycle
     if(igy < 1 .or. igy > ny) cycle
     if(igz < 1 .or. igz > nz) cycle

! B ----------------------------
     nvalB(igx,igy,igz,idta)=nvalB(igx,igy,igz,idta) +1

     gridB(1,igx,igy,igz,idta)=gridB(1,igx,igy,igz,idta) +Bx(iclu)  
     gridB(2,igx,igy,igz,idta)=gridB(2,igx,igy,igz,idta) +By(iclu)  
     gridB(3,igx,igy,igz,idta)=gridB(3,igx,igy,igz,idta) +Bz(iclu)  
     
     B0=sqrt(Bx(iclu)**2 +By(iclu)**2 +Bz(iclu)**2)

     gridB(4,igx,igy,igz,idta)=gridB(4,igx,igy,igz,idta) +B0
     
! J --------------------------------

! **************************************
    if(Dmax  > 1.88) cycle  ! 12000 km =1.88 RE;  10000 km =1.57 RE
    if(elong > 0.8)  cycle
    if(plana > 0.8)  cycle
! **************************************
  
    nvalJ(igx,igy,igz,idta)=nvalJ(igx,igy,igz,idta) +1

    gridJ(1,igx,igy,igz,idta)=gridJ(1,igx,igy,igz,idta) +GJx  
    gridJ(2,igx,igy,igz,idta)=gridJ(2,igx,igy,igz,idta) +GJy  
    gridJ(3,igx,igy,igz,idta)=gridJ(3,igx,igy,igz,idta) +GJz  
  
    G0=sqrt(GJx**2 +GJy**2 +GJz**2)
    gridJ(4,igx,igy,igz,idta)=gridJ(4,igx,igy,igz,idta) +G0

    gridJ(5,igx,igy,igz,idta)=gridJ(5,igx,igy,igz,idta) +abs(div)
    if(curl > 0.) gridJ(6,igx,igy,igz,idta)=gridJ(6,igx,igy,igz,idta) +abs(div/curl)
    gridJ(7,igx,igy,igz,idta)=gridJ(7,igx,igy,igz,idta) +abs(ajb)
    
  end do ! sur iclu
  
  ENDDO ! sur tous les points du fichier


 20 continue  ! pour EOF sur fichier en cours

  print 100, '     ',trim(file_curl_div),'  nr   =',nr

  ENDDO ! sur lecture de tous les fichiers
  
 30 continue ! tous les fichiers sont lus

  print*
  print*, '------------------------------------------------'
  print*, 'All ',nf,' files are read'
  print*, 'Total number of point in the grid: ',nt
  print*, '------------------------------------------------'
  print*
  print*
  print*,'-----------------------------------------------------------------'
  print*,' 2) grids creation for various values of dip. tilt angle, wait...'
  print*,'-----------------------------------------------------------------'
  print*
  
! creation des fichiers grilles pour chaque valeur du dta
! -------------------------------------------------------

do idta=1,nbdta

  dta=float(idta-5)*10.
  kdta=INT(dta)
  write(cdta,'(i3.2)') abs(kdta)
  if(kdta >= 0) then
          cdta(1:1)='p'
                else
          cdta(1:1)='m'
  endif
  print*, 'grids for dip tilt angle idta,cdta=',idta,cdta

  write(datafileB, '(a,3i3.3,3a)') trim(dirgrid)//'/grid_Bxyz_',nx,ny,nz,'_',cdta,'.dat'
  write(datafileN, '(a,3i3.3,3a)') trim(dirgrid)//'/grid_Nxyz_',nx,ny,nz,'_',cdta,'.dat'
  write(datafileJ, '(a,3i3.3,3a)') trim(dirgrid)//'/grid_Jxyz_',nx,ny,nz,'_',cdta,'.dat'

  open(11,file=datafileB)
  open(12,file=datafileN)
  open(13,file=datafileJ)

  write(11,'(i8,1x,i8,1x,i8,1x,i1)') nx,ny,nz,4
  write(12,'(i8,1x,i8,1x,i8,1x,i1)') nx,ny,nz,1
  write(13,'(i8,1x,i8,1x,i8,1x,i1)') nx,ny,nz,7 

  nb=0
  nj=0     

! normalisation

  do i=1,nx
     do j=1,ny
        do k=1,nz
          if(nvalB(i,j,k,idta) /= 0) then
             Bxx=gridB(1,i,j,k,idta)/float(nvalB(i,j,k,idta))
             Byy=gridB(2,i,j,k,idta)/float(nvalB(i,j,k,idta))
             Bzz=gridB(3,i,j,k,idta)/float(nvalB(i,j,k,idta))          
             B00=gridB(4,i,j,k,idta)/float(nvalB(i,j,k,idta))
             nb=nb+1
                              else
             Bxx=1.e30
             Byy=1.e30
             Bzz=1.e30
             B00=1.e30
          endif
          
          write(11,'(4E13.5,i8)') Bxx,Byy,Bzz,B00, nvalB(i,j,k,idta)
          write(12,'( E13.5,i8)') Float(nvalB(i,j,k,idta)), nvalB(i,j,k,idta)

          if(nvalJ(i,j,k,idta) /= 0) then
             Jxx=gridJ(1,i,j,k,idta)/float(nvalJ(i,j,k,idta))
             Jyy=gridJ(2,i,j,k,idta)/float(nvalJ(i,j,k,idta))
             Jzz=gridJ(3,i,j,k,idta)/float(nvalJ(i,j,k,idta))
             J00=gridJ(4,i,j,k,idta)/float(nvalJ(i,j,k,idta))

             Div=gridJ(5,i,j,k,idta)/float(nvalJ(i,j,k,idta))
             Rap=gridJ(6,i,j,k,idta)/float(nvalJ(i,j,k,idta))
             Ang=gridJ(7,i,j,k,idta)/float(nvalJ(i,j,k,idta))

             nj=nj+1
                              else
             Jxx=1.e30
             Jyy=1.e30
             Jzz=1.e30
             J00=1.e30
             Div=1.e30
             Rap=1.e30
             Ang=1.e30
          endif

          write(13,'(7E13.5,i8)') Jxx,Jyy,Jzz,J00,Div,Rap,Ang, nvalJ(i,j,k,idta)

        enddo
     enddo
  enddo

  print*
  print*, '---------------------------------------------------'
  print*,trim(datafileB)
  print*,trim(datafileN)
  print*,trim(datafileJ)

  print*,'grid dimension        :',nx,ny,nz
  print*,'nb pts for B grid :',nb
  print*,'nb pts for N grid :',nb
  print*,'nb pts for J grid :',nj
  print*, '---------------------------------------------------'
  
  close(11)
  close(12)
  close(13)

enddo ! idta

  print*, ('-',i=1,72)

  print*, "    crea_3D_grid.exe  : NORMAL TERMINATION"
         stop "crea_3D_grid.exe  : NORMAL TERMINATION"

  end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine codist(satpos,dmin,dmax)
      
      real satpos(3,4)
!
!     ------------------------------------------------------------------
      common /distan/     d12 ,    d13 ,    d14 ,    d23 ,  d24,  d34, &
                         dx12 ,   dx13 ,   dx14 ,   dx21 , dx31, dx41, &
                         dy12 ,   dy13 ,   dy14 ,   dy21 , dy31, dy41, &
                         dz12 ,   dz13 ,   dz14 ,   dz21 , dz31, dz41
!     ------------------------------------------------------------------

      call  cdistance(satpos)
      
      dmin=min(d12 ,d13 ,d14 ,d23 ,d24,d34)
      dmax=max(d12 ,d13 ,d14 ,d23 ,d24,d34)
      
      return
      end
! XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX      


